/*jshint esversion: 6 */

define(["immutable", "src/math/mathUtils", "src/math/Mat3", "lodash"],
function(immutable, mathUtils, mat3, lodash) {
"use strict";

var exports = {};

/*============================================================================
	Matrix State
============================================================================*/

/** matrixType enumerates four ways of controlling the Matrix dofs:
 * 0 = 00 = { linear : false, translation : false } = kAffine
 * 1 = 01 = { linear : false, translation : true } = kLinear
 * 2 = 10 = { linear : true, translation : false } = kTranslation
 * 3 = 11 = { linear : true, translation : true } = kNotSet
 */

var	kAffine = 0,
	kLinear = 1,
	kTranslation = 2,
	kNotSet = 3;

var type = {
	kAffine,
	kLinear,
	kTranslation,
	kNotSet,

	merge (a, b) {
		/* jshint bitwise: false */
		return a & b;
	},

	propagate (a) {
		if (a < kTranslation) return kAffine;
		if (a > kTranslation) return kNotSet;

		return kTranslation;
	},

	fromAutomation (auto) {
		var result = 0;
		if (auto.translation) result += 1;
		if (auto.linear) result += 2;
		return result;
	}
};
exports.type = type;

/**
 * Matrix dof is a record describing type (matrixType) and value (mat3).
 */
var Matrix = immutable.Record({ 
	// defaults to unset matrix
	type: type.kNotSet, 

	// defaults to identity matrix so that it has no effect in matrix operations
	// that way the value can be safely used in all matrix operations (as a noop) even when its not set.
	matrix : mat3().setIdentity() 
}, "MatrixDof");
exports.Matrix = Matrix;

/**
 * A List of Matrix dofs (LoM) represents the state of every layer *without* sublayers.
 * LoM is implemented with an immutable data type:
 *
 * immutable.List<Matrix>
 */

/**
 * Tree<Any>
 */
var Tree = immutable.Record({
	// list of dofs representing the state of this layer.
 	value : immutable.List(), 		// List<Any>
 	// Map of trees with each entry representing the state of each sub-layer.
 	children : immutable.Map(), 	// Map<Tree>
}, "Tree");
exports.Tree = Tree;

 /**
 * A Tree of Matrix dofs (ToM) represents the state of a layer *with* sublayers.
 * ToM is implemented with an immutable data type: 
 *
 * Tree<Matrix>
 */


function isTree (tree) {
	return tree instanceof Tree;
}
exports.isTree = isTree;

// Sets key/value pair for an Immutable object only if the new value is not eq.
function setOnChange (destination, key, prev, next) {
	return (prev === next) ? destination : destination.set(key, next);
}
exports.setOnChange = setOnChange;

// Invokes updater with current value and sets only when it returns another value.
// The updater function can return its input argument when it wants no change.
function updateOnChange (destination, key, updater) {
	var prev = destination.get(key);
	return setOnChange(destination, key, prev, updater(prev));
}
exports.updateOnChange = updateOnChange;

// Minimizes matrix dof changes but retaining old value when it's not appreciable different from the new value.
function updateOnMatrixChange (prev, next) {
	if (prev.get("type") !== next.get("type")) return next;
	if (!prev.get("matrix").equalsApproximately(next.get("matrix"))) return next;

	return prev;
}
exports.updateOnMatrixChange = updateOnMatrixChange;

// Recursive (deep) update of all 'value' keys in a Tree<X>.
// Given a current value, the updater function (X -> X) returns a new value.
function updateValues (withUpdater, forest) {
	var withDeepUpdater = lodash.partial(updateValues, withUpdater);
	return forest.withMutations(function (forest) {
		if (forest.isEmpty()) return;
		if (isTree(forest)) {
			let tree = forest;
			updateOnChange(tree, "value", withUpdater);
			updateOnChange(tree, "children", withDeepUpdater);
		} else {
			forest.forEach(function (prev, key) {
				forest = setOnChange(forest, key, prev, withDeepUpdater(prev));
			});
		}
	});
}
Tree.updateValues = updateValues;

var getTreeValue = lodash.method("get", "value"),
	getTreeChildren = lodash.method("get", "children");

// Recursive (deep) update of all 'value' keys in a Tree<X>, 
// while simultaneously walking structurally identical Tree<Y>, Tree<Z>, ...
// Given current values in each tree, the updater function (X Y Z ... -> X) returns a new value.
function updateValuesWithZipped (updater, first, ...rest) {
	return first.withMutations(function (first) {
		if (first.isEmpty()) return;
		if (isTree(first)) {
			let tree = first;
			updateOnChange(tree, "value", function (valFirst) { 
				var valRest = rest.map(getTreeValue);
				return updater(valFirst, ...valRest);
			});
			updateOnChange(tree, "children", function (forestFirst) {
				var forestRest = rest.map(getTreeChildren);
				return updateValuesWithZipped(updater, forestFirst, ...forestRest);
			});
		} else {
			let forest = first;
			forest.forEach(function (treeFirst, key) {
				var treeRest = rest.map(f => f.get(key));
				setOnChange(forest, key, treeFirst, updateValuesWithZipped(updater, treeFirst, ...treeRest));
			});
		}
	});
}
Tree.updateValuesWithZipped = updateValuesWithZipped;

// Recursive (deep) fold of a Tree<X> into single value of type R,
// while simultaneously walking structurally identical Tree<Y>, Tree<Z>, ...
// fnInit: () -> R
// fnForest: fnInit R -> R
// fnTree: R X Y Z ... -> R
function fold (fnTree, fnForest, fnInit, first, ...rest) {
	if (first.isEmpty()) return fnInit();

	if (isTree(first)) {
		var forestFirst = getTreeChildren(first),
			forestRest = rest.map(getTreeChildren);

		return fnTree(
			fold(fnTree, fnForest, fnInit, forestFirst, ...forestRest),
			getTreeValue(first),
			...rest.map(getTreeValue)
		);
	}

	var folded = first.map(function (treeFirst, treeKey) {
		var treeRest = rest.map((forest) => forest.get(treeKey));
		return fold(fnTree, fnForest, fnInit, treeFirst, ...treeRest);
	});

	return fnForest(fnInit, folded);
}

function accumulateTreeDiffs (folded, valFirst, valSecond) {
	if (valFirst !== valSecond) {
		folded["/value"] = [ valFirst, valSecond ];
	}
	return folded;
}

function accumulateForestDiffs (fnInit, forestDiffs) {
	return forestDiffs.reduce((folded, treeDiffs, treeKey) => {
		lodash.forOwn(treeDiffs, (diff, diffKey) => {
			folded["/" + treeKey + diffKey] = diff;
		});
		return folded;
	}, fnInit());
}

function initDiffs () {
	return {};
}

// Diff Tree<X> and Tree<Y> and return an object with key/value pairs for each difference.
// The key string identifies the path in the tree and the value tuple contains first and second value.
function diff (first, second) {
	return fold(accumulateTreeDiffs, accumulateForestDiffs, initDiffs, first, second);
}
Tree.diff = diff;

/*============================================================================
	Transform State
============================================================================*/

/**
 * A List of Transform dofs (LoT) and Tree of Transform dofs (ToT) offer another representation of Layer state.
 * Although Transform dofs ultimately map back to Matrix dofs, they enable richer composition (i.e. rotation winding)
 * with seperate component for translation, rotation, scale, and so on.
 * 
 * LoT and ToT implementations are also backed by immutable data types:
 * immutable.List<TransformDof> and Tree<TransformDof>.
 *
 * NOTE: Transform is plain JS object, not an immutable type.  Immutable data operations will make shallow copies:
 * *reference* copy not value copy.
 *
 */

/**
 * Transform is plain (mutable) JS object.
 */

var kTransformDefault = {
	type : type.kNotSet,

	x : 0,
	y : 0,
	angle : 0,
	xScale : 1,
	yScale : 1,
	xShear : 0
};
exports.TransformDefault = kTransformDefault;

function Transform (args0) {
	if (args0) {
		lodash.defaults(this, args0, kTransformDefault);
	} else {
		lodash.defaults(this, kTransformDefault);
	}
}
exports.Transform = Transform;

const positionTmp = [],
	scaleTmp = [],
	shearTmp = [],
	rotationTmp = [];

function transformFromMatrix (mdof)
{
	var type = mdof.get("type"),
		mat = mdof.get("matrix");

	mat3.decomposeAffine(mat, positionTmp, scaleTmp, shearTmp, rotationTmp);

	return new Transform({
		type,
		x : positionTmp[0],
		y : positionTmp[1],
		angle : mathUtils.angle(rotationTmp),
		xScale : scaleTmp[0],
		yScale : scaleTmp[1],
		xShear : shearTmp[0]
	});
}
exports.transformFromMatrix = transformFromMatrix;

function matFromTransform (tdof, result0) {
	positionTmp[0] = tdof.x; 
	positionTmp[1] = tdof.y;

	scaleTmp[0] = tdof.xScale; 
	scaleTmp[1] = tdof.yScale;

	shearTmp[0] = tdof.xShear;

	return mat3.affine(positionTmp, scaleTmp, shearTmp, tdof.angle, result0);
}
exports.matFromTransform = matFromTransform;

function matrixFromTransform (tdof) 
{
	return new Matrix({
		type : tdof.type,
		matrix : matFromTransform(tdof)
	});
}
exports.matrixFromTransform = matrixFromTransform;

function typeFromKey (key) {
	if (key === "x" || key === "y")	return type.kTranslation;

	return type.kLinear;
}
exports.typeFromKey = typeFromKey;

return exports;
});  // end define